// YoutubeTS Chrome Extension - Content Script
// Injects "Video To Text" button on YouTube video pages

(function() {
    'use strict';

    let buttonInjected = false;
    let currentVideoId = null;

    // Extract video ID from YouTube URL
    function getVideoId() {
        const urlParams = new URLSearchParams(window.location.search);
        const videoId = urlParams.get('v');
        return videoId;
    }

    // Create and inject the "Video To Text" button
    function injectButton() {
        // Check if we're on a video page
        const videoId = getVideoId();
        if (!videoId) {
            buttonInjected = false;
            return;
        }

        // Don't inject if already injected for this video
        if (buttonInjected && currentVideoId === videoId) {
            // Double check button still exists in DOM
            const existingButton = document.getElementById('youtubets-button');
            if (existingButton) {
                return;
            }
        }

        // Remove old button/wrapper if video changed
        const existingWrapper = document.getElementById('youtubets-button-wrapper');
        if (existingWrapper) {
            existingWrapper.remove();
        }
        const existingButton = document.getElementById('youtubets-button');
        if (existingButton) {
            existingButton.remove();
        }

        // Try multiple times with decreasing intervals for faster detection
        let attempts = 0;
        const maxAttempts = 50; // Try for up to 10 seconds (50 * 200ms)

        const checkInterval = setInterval(() => {
            attempts++;

            // Find the exact container - try multiple selectors
            const buttonContainer =
                document.querySelector('ytd-menu-renderer.ytd-watch-metadata #top-level-buttons-computed') ||
                document.querySelector('#top-level-buttons-computed') ||
                document.querySelector('ytd-menu-renderer #top-level-buttons-computed');

            if (buttonContainer) {
                clearInterval(checkInterval);

                // Create button wrapper that matches YouTube's button style
                const wrapper = document.createElement('div');
                wrapper.id = 'youtubets-button-wrapper';
                wrapper.style.cssText = 'display: inline-flex; align-items: center;';

                // Create the button using YouTube's button structure
                const button = document.createElement('button');
                button.id = 'youtubets-button';
                button.className = 'yt-spec-button-shape-next yt-spec-button-shape-next--tonal yt-spec-button-shape-next--mono yt-spec-button-shape-next--size-m';
                button.style.cssText = `
                    background: linear-gradient(135deg, #FF0055, #6C5CE7) !important;
                    color: white !important;
                    border: none !important;
                    border-radius: 18px !important;
                    padding: 0 16px !important;
                    height: 36px !important;
                    font-size: 14px !important;
                    font-weight: 500 !important;
                    cursor: pointer !important;
                    transition: all 0.2s !important;
                    margin-right: 8px !important;
                    display: flex !important;
                    align-items: center !important;
                    gap: 8px !important;
                `;

                button.innerHTML = `
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="flex-shrink: 0;">
                        <path d="M4 4h16v2H4V4zm0 4h16v2H4V8zm0 4h16v2H4v-2zm0 4h10v2H4v-2z" fill="white"/>
                    </svg>
                    <span style="white-space: nowrap;">Video To Text</span>
                `;

                // Add hover effect
                button.addEventListener('mouseenter', () => {
                    button.style.background = 'linear-gradient(135deg, #ff1a6b, #7d6df0) !important';
                    button.style.transform = 'translateY(-1px)';
                    button.style.boxShadow = '0 4px 12px rgba(255, 0, 85, 0.5)';
                });

                button.addEventListener('mouseleave', () => {
                    button.style.background = 'linear-gradient(135deg, #FF0055, #6C5CE7) !important';
                    button.style.transform = 'translateY(0)';
                    button.style.boxShadow = 'none';
                });

                // Add click event - open popup window
                button.addEventListener('click', (e) => {
                    e.preventDefault();
                    e.stopPropagation();

                    const currentVideoId = getVideoId();
                    if (currentVideoId) {
                        const url = `https://youtubets.com/watch?v=${currentVideoId}`;

                        // Open in popup window with optimal size
                        const width = 1200;
                        const height = 800;
                        const left = (screen.width - width) / 2;
                        const top = (screen.height - height) / 2;

                        window.open(
                            url,
                            'YoutubeTS',
                            `width=${width},height=${height},left=${left},top=${top},resizable=yes,scrollbars=yes,status=yes`
                        );
                    }
                });

                wrapper.appendChild(button);

                // Insert at the beginning of the button container
                buttonContainer.insertBefore(wrapper, buttonContainer.firstChild);

                buttonInjected = true;
                currentVideoId = videoId;
            } else if (attempts >= maxAttempts) {
                // Stop trying after max attempts
                clearInterval(checkInterval);
                console.log('YoutubeTS: Failed to find button container after', maxAttempts, 'attempts');
            }
        }, 200); // Check every 200ms for faster response
    }

    // Observe URL changes (YouTube is a SPA - Single Page Application)
    function observeUrlChange() {
        let lastUrl = location.href;

        new MutationObserver(() => {
            const url = location.href;
            if (url !== lastUrl) {
                lastUrl = url;
                buttonInjected = false;
                currentVideoId = null;

                // Inject immediately when URL changes
                setTimeout(() => {
                    injectButton();
                }, 100); // Reduced from 1000ms to 100ms
            }
        }).observe(document, { subtree: true, childList: true });
    }

    // Watch for YouTube's specific navigation events
    function observeYoutubeNavigation() {
        // Listen for yt-navigate-finish event (YouTube's SPA navigation)
        document.addEventListener('yt-navigate-finish', () => {
            buttonInjected = false;
            currentVideoId = null;
            setTimeout(() => {
                injectButton();
            }, 100);
        });

        // Listen for video data loaded event
        document.addEventListener('yt-page-data-updated', () => {
            if (!buttonInjected || !document.getElementById('youtubets-button')) {
                setTimeout(() => {
                    injectButton();
                }, 100);
            }
        });
    }

    // Initialize
    function init() {
        // Immediate first attempt
        injectButton();

        // Second attempt after short delay
        setTimeout(() => {
            injectButton();
        }, 500);

        // Third attempt after moderate delay
        setTimeout(() => {
            injectButton();
        }, 1500);

        // Watch for URL changes
        observeUrlChange();

        // Watch for YouTube navigation events
        observeYoutubeNavigation();

        // Aggressive re-check every 2 seconds (backup in case button gets removed)
        setInterval(() => {
            const videoId = getVideoId();
            if (videoId && !document.getElementById('youtubets-button')) {
                injectButton();
            }
        }, 2000); // Reduced from 3000ms to 2000ms
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
